<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

use App\Models\Docente;
use App\Models\Usuario;
use App\Models\Curso;
use App\Models\CursoDocente;
use App\Models\Role;
use App\Models\Area;
use Illuminate\Support\Facades\Hash;
class DocenteController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $docentes = Docente::with('usuario.roles','usuario.areas')->get();
        $roles = Role::where('estado', 1)->get(); // Solo roles activos
        $areas = Area::where('estado', 'Activo')->get(); // Áreas activas

        return view('docentes.index', compact('docentes', 'roles', 'areas'));
    }


    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
        $roles = Role::where('estado', 1)->get();
        return view('docentes.create', compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store1(Request $request)
    {
        // Validación de los datos del formulario
        $request->validate([
            'nombre' => 'required|string|max:100',
            'apellido' => 'required|string|max:100',
            'dni' => 'required|string|max:15|unique:docentes,dni|unique:usuarios,username',  // Validación de DNI único
            'email' => 'nullable|email|max:100',
            'telefono' => 'nullable|string|max:20',
            'direccion' => 'nullable|string|max:255',
            'departamento' => 'nullable|string|max:100',
            'cargo' => 'nullable|string|max:50',
            'roles' => 'required|array', // El campo roles debe ser un array
            'roles.*' => 'exists:roles,id', // Cada rol debe existir en la base de datos
            'areas' => 'nullable|array', // Campo áreas (opcional)
            'areas.*' => 'exists:areas,id', // Cada área debe existir en la base de datos
        ]);

        // Verificar si el DNI ya existe en la tabla docentes o usuarios
        $existingDocente = Docente::where('dni', $request->dni)->first();
        $existingUsuario = Usuario::where('username', $request->dni)->first();

        if ($existingDocente || $existingUsuario) {
            return response()->json(['error' => 'El DNI ya está registrado en el sistema.'], 400);
        }

        // Crear usuario
        $usuario = Usuario::create([
            'username' => $request->dni, // DNI como username
            'password_hash' => Hash::make($request->dni), // DNI como contraseña
            'status' => 'Disponible', // Estado por defecto
        ]);

        // Asignar roles al usuario
        $usuario->roles()->attach($request->roles);

        // Asignar áreas al usuario si existen
        if ($request->has('areas')) {
            $usuario->areas()->attach($request->areas);
        }

        // Crear docente
        $docente = Docente::create([
            'usuario_id' => $usuario->id,
            'nombre' => $request->nombre,
            'apellido' => $request->apellido,
            'dni' => $request->dni,
            'email' => $request->email,
            'telefono' => $request->telefono,
            'direccion' => $request->direccion,
            'departamento' => $request->departamento,
            'cargo' => $request->cargo,
        ]);

        return response()->json(['success' => 'Docente registrado correctamente.'], 200);
    }
public function store(Request $request, \App\Services\MoodleService $moodle)
{
    // Validación
    $request->validate([
        'nombre' => 'required|string|max:100',
        'apellido' => 'required|string|max:100',
        'dni' => 'required|string|max:15|unique:docentes,dni|unique:usuarios,username',
        'email' => 'nullable|email|max:100',
        'telefono' => 'nullable|string|max:20',
        'direccion' => 'nullable|string|max:255',
        'departamento' => 'nullable|string|max:100',
        'cargo' => 'nullable|string|max:50',
        'roles' => 'required|array',
        'roles.*' => 'exists:roles,id',
        'areas' => 'nullable|array',
        'areas.*' => 'exists:areas,id',
    ]);

    // 1. Crear usuario del sistema
    $usuario = Usuario::create([
        'username' => $request->dni,
        'password_hash' => Hash::make($request->dni),
        'status' => 'Disponible',
    ]);

    // Asignar roles locales
    $usuario->roles()->attach($request->roles);

    if ($request->has('areas')) {
        $usuario->areas()->attach($request->areas);
    }

    // 2. Crear docente
    $docente = Docente::create([
        'usuario_id' => $usuario->id,
        'nombre' => $request->nombre,
        'apellido' => $request->apellido,
        'dni' => $request->dni,
        'email' => $request->email,
        'telefono' => $request->telefono,
        'direccion' => $request->direccion,
        'departamento' => $request->departamento,
        'cargo' => $request->cargo,
    ]);

    // ==========================================================
    // 3. CREAR USUARIO EN MOODLE
    // ==========================================================

    $passwordMoodle = "Docente@" . $request->dni;

    $resp = $moodle->crearUsuario(
        $usuario->username,
        $passwordMoodle,
        $docente->nombre,
        $docente->apellido,
        $docente->email ?? ($usuario->username . "@noemail.com")
    );

    if (isset($resp[0]['id'])) {
        $usuario->moodle_user_id = $resp[0]['id'];
        $usuario->save();
    }

    return response()->json(['success' => 'Docente registrado correctamente y sincronizado con Moodle.']);
}



    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $roles = Role::where('estado', 1)->get();
        return view('docentes.edit', compact('docente', 'roles'));
    }


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
{
    $docente = Docente::findOrFail($id);

    $request->validate([
        'nombre' => 'required|string|max:100',
        'apellido' => 'required|string|max:100',
        'email' => 'nullable|email|max:100',
        'telefono' => 'nullable|string|max:20',
        'direccion' => 'nullable|string|max:255',
        'departamento' => 'nullable|string|max:100',
        'cargo' => 'nullable|string|max:50',
        'roles' => 'array',
        'areas' => 'array', // Validar que llegue como array
    ]);

    // Actualizar datos del docente
    $docente->update($request->only([
        'nombre', 'apellido', 'email', 'telefono', 'direccion', 'departamento', 'cargo'
    ]));

    // Obtener el usuario asociado
    $usuario = $docente->usuario;

    // Actualizar roles
    $rolesSeleccionados = $request->input('roles', []);
    $usuario->roles()->sync($rolesSeleccionados);

    // Actualizar áreas (desde el modelo Usuario, NO Docente)
    $areasSeleccionadas = $request->input('areas', []);
    $usuario->areas()->sync($areasSeleccionadas); // ← Esto es correcto

    return response()->json(['success' => true, 'message' => 'Docente actualizado correctamente.']);
}



    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            // Buscar el docente
            $docente = Docente::findOrFail($id);

            // Verificar si el docente tiene un usuario asociado
            if ($docente->usuario) {
                // 1. Eliminar las relaciones de roles asociados al usuario
                $docente->usuario->roles()->detach();

                // 2. Eliminar la relación de usuario_id en la tabla docentes
                $docente->usuario_id = null; // Desasociar el usuario
                $docente->save();

                // 3. Eliminar el usuario
                $docente->usuario->delete();
            }

            // 4. Eliminar el docente
            $docente->delete();

            // Retornar respuesta exitosa
            return response()->json(['success' => 'Docente y su usuario eliminados correctamente.']);
        } catch (\Exception $e) {
            // Capturar el error y retornarlo para depuración
            return response()->json(['error' => 'Ocurrió un error al eliminar el docente. ' . $e->getMessage()], 500);
        }
    }

    


    public function getRoles($id)
{
    // Buscar el docente por su ID, con roles y áreas del usuario relacionados
    $docente = Docente::with('usuario.roles', 'usuario.areas')->findOrFail($id);

    // Obtener solo los IDs de los roles y áreas que tiene el docente
    $rolesDelDocente = $docente->usuario->roles->pluck('id')->toArray();
    $areasDelDocente = $docente->usuario->areas->pluck('id')->toArray();

    // Obtener todos los roles y áreas disponibles
    $roles = Role::all();
    $areas = Area::all();

    // Retornar roles, áreas y los asignados para el docente
    return response()->json([
        'roles' => $roles,
        'rolesDelDocente' => $rolesDelDocente,
        'areas' => $areas,
        'areasDelDocente' => $areasDelDocente,
    ]);
}


    public function informacion($id)
    {
        // Validar acceso del docente
        $usuario = Auth::user();
        $docente = Docente::where('usuario_id', $usuario->id)->first();
        $cursoDocente = CursoDocente::where('curso_id', $id)->where('docente_id', $docente->id)->first();

        if (!$cursoDocente) {
            return redirect()->route('dashboard')->with('error', 'No tienes acceso a este curso.');
        }

        // Obtener datos básicos del curso
        $curso = Curso::findOrFail($id);

        return view('docentes.cursos.informacion', compact('curso'));
    }

    public function verPerfil()
    {
        $docente = Auth::user()->docente;

        if (!$docente) {
            return redirect()->route('home')->with('error', 'Docente no autenticado.');
        }

        return view('docentes.perfil', compact('docente'));
    }

    public function actualizarClave(Request $request)
    {
        $request->validate([
            'password' => 'required|min:6|confirmed',
        ]);

        $usuario = Auth::user();
        $usuario->password_hash = Hash::make($request->password);
        $usuario->save();

        return redirect()->route('docente.perfil')->with('success', 'Contraseña actualizada correctamente.');
    }

    public function actualizarPerfil(Request $request)
    {
        $docente = Auth::user()->docente;
        $usuario = Auth::user();

        $request->validate([
            'nombre' => 'required|string|max:100',
            'apellido' => 'required|string|max:100',
            'email' => 'required|email',
            'telefono' => 'nullable|string|max:15',
            'direccion' => 'nullable|string|max:100',
            
            'password' => 'nullable|min:6|confirmed',
        ]);

        $docente->update($request->only([
            'nombre', 'apellido', 'email', 'telefono', 'direccion', 'departamento', 'cargo'
        ]));

        if ($request->filled('password')) {
            $usuario->password_hash = Hash::make($request->password);
            $usuario->save();
        }

        return response()->json(['message' => 'Perfil actualizado correctamente.']);
    }

public function importarExcel(Request $request)
{
    $request->validate([
        'archivo' => 'required|mimes:xlsx,xls'
    ]);

    try {
        \Excel::import(new \App\Imports\DocentesMasivoImport, $request->file('archivo'));

        return response()->json([
            'success' => true,
            'message' => 'Los docentes fueron importados correctamente.'
        ]);

    } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {

        $errores = [];
        foreach ($e->failures() as $failure) {
            $errores[] = "Fila {$failure->row()}: " . implode(", ", $failure->errors());
        }

        return response()->json(['errors' => $errores], 422);

    } catch (\Exception $e) {

        return response()->json([
            'success' => false,
            'message' => "Error: " . $e->getMessage()
        ], 500);
    }
}


}
