<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ArchivoCurso;
use App\Models\Curso;
use App\Models\Sesion;
use App\Models\CursoDocente;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class ArchivoCursoController extends Controller
{
    public function index($curso_id)
{
    $docente_id = Auth::user()->docente->id ?? null;

    // Verificar si el curso pertenece al docente
    $asignado = CursoDocente::where('curso_id', $curso_id)
        ->where('docente_id', $docente_id)
        ->exists();

    if (!$asignado) {
        abort(403, 'No tienes permiso para acceder a este curso.');
    }

    $curso = Curso::findOrFail($curso_id);

    // Cargar archivos con su sesión (si existe)
    $archivos = ArchivoCurso::with('sesion')
        ->where('curso_id', $curso_id)
        ->orderByDesc('created_at')
        ->get();

    // Obtener las sesiones del curso para el modal
    $sesiones = Sesion::where('curso_id', $curso_id)
        ->orderByDesc('fecha')
        ->get();

    return view('docentes.archivos.index', compact('archivos', 'curso', 'sesiones'));
}


    public function store(Request $request, $curso_id)
{
    $request->validate([
        'nombre' => 'required|string|max:255',
        'tipo' => 'required|in:archivo,video',
        'archivo' => 'required_if:tipo,archivo|file|max:10240',
        'ruta_video' => 'required_if:tipo,video|url',
        'sesion_id' => 'nullable|exists:sesiones,id_sesion'
    ]);

    $ruta = null;

    if ($request->tipo === 'archivo') {
        $archivoSubido = $request->file('archivo');
        $nombreArchivo = time() . '_' . $archivoSubido->getClientOriginalName();

        $directorio = public_path('archivos');
        if (!file_exists($directorio)) {
            mkdir($directorio, 0755, true);
        }

        $archivoSubido->move($directorio, $nombreArchivo);
        $ruta = 'archivos/' . $nombreArchivo;
    } else {
        $ruta = $request->ruta_video;
    }

    $archivo = new ArchivoCurso([
        'curso_id' => $curso_id,
        'tipo' => $request->tipo,
        'nombre' => $request->nombre,
        'ruta' => $ruta,
        'sesion_id' => $request->sesion_id,
        'fecha_subida' => now()
    ]);

    $archivo->save();

    return response()->json(['success' => true, 'message' => 'Archivo subido correctamente.']);
}





    public function update(Request $request, $id)
    {
        $archivo = ArchivoCurso::findOrFail($id);

        $request->validate([
            'nombre' => 'required|string|max:255',
            'tipo' => 'required|in:archivo,video',
            'archivo' => 'nullable|file|max:10240',
        ]);

        // Manejo de archivo si se sube uno nuevo
        if ($request->hasFile('archivo')) {
            // Eliminar archivo anterior si existe
            if ($archivo->ruta && Storage::exists('public/' . $archivo->ruta)) {
                Storage::delete('public/' . $archivo->ruta);
            }

            $archivoSubido = $request->file('archivo');
            $nombreArchivo = time() . '_' . $archivoSubido->getClientOriginalName();
            $ruta = $archivoSubido->storeAs('archivos_curso', $nombreArchivo, 'public');
        } else {
            $ruta = $archivo->ruta; // mantener la ruta antigua
        }

        $archivo->update([
            'nombre' => $request->nombre,
            'tipo' => $request->tipo,
            'ruta' => $ruta,
        ]);

        return response()->json(['success' => true, 'message' => 'Archivo actualizado correctamente.']);
    }

    public function destroy($id)
    {
        $archivo = ArchivoCurso::findOrFail($id);

        // Si es tipo archivo y existe físicamente, lo eliminamos
        if ($archivo->tipo === 'archivo') {
            $rutaFisica = public_path($archivo->ruta);
            if (file_exists($rutaFisica)) {
                unlink($rutaFisica);
            }
        }

        $archivo->delete();

        return response()->json(['message' => 'Archivo eliminado correctamente.']);
    }

}
