<?php

namespace App\Http\Controllers;

use App\Models\Curso;
use App\Models\PlanEstudio;
use App\Models\Semestre;
use App\Models\ModuloFormativo;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CursoController extends Controller
{
    public function index(Request $request)
    {
        // Consulta base con relaciones
        $cursos = Curso::with(['modulo', 'semestre', 'cursosPlanEstudios.planEstudio']);

        // Filtrar por nombre
        if ($request->filled('nombre')) {
            $cursos->where('nombre', 'like', '%' . $request->nombre . '%');
        }

        // Filtrar por módulo formativo
        if ($request->filled('modulo_id')) {
            $cursos->where('id_modulo', $request->modulo_id);
        }

        // Filtrar por semestre
        if ($request->filled('semestre_id')) {
            $cursos->where('semestre_id', $request->semestre_id);
        }

        // Filtrar por plan de estudio (usando la tabla intermedia)
        if ($request->filled('plan_estudio_id')) {
            $cursos->whereHas('cursosPlanEstudios', function ($query) use ($request) {
                $query->where('plan_estudio_id', $request->plan_estudio_id);
            });
        }

        // Ejecutar consulta
        $cursos = $cursos->get();

        if ($request->ajax()) {
            return view('cursos.partials.tabla', compact('cursos'))->render();
        }

        // Datos para los select
        $planesEstudio = PlanEstudio::all();
        $semestres = Semestre::all();
        $modulos = ModuloFormativo::all();

        return view('cursos.index', compact('cursos', 'planesEstudio', 'semestres', 'modulos'));
    }

    // Mostrar el formulario para crear un nuevo curso
    public function create()
    {
        // Obtener todos los planes de estudio, semestres y módulos formativos para el formulario
        $planesEstudio = PlanEstudio::all();
        $semestres = Semestre::all();
        $modulos = ModuloFormativo::all();

        return view('cursos.create', compact('planesEstudio', 'semestres', 'modulos'));
    }


    // Almacenar un nuevo curso en la base de datos
    public function store1(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:100',
            'descripcion' => 'nullable|string',
            'tipo' => 'required|string',
            'creditos' => 'nullable|string|max:3',
            'horas_semestrales' => 'nullable|integer',
            'orden' => 'nullable|integer',
            'semestre_id' => 'required|exists:semestres,id',
            'id_modulo' => 'required|exists:modulos_formativos,id_modulo',
            'planes_estudio' => 'required|array',
            'planes_estudio.*' => 'exists:planes_estudio,id',
        ]);

        try {
            $curso = new Curso();
            $curso->nombre = $request->nombre;
            $curso->descripcion = $request->descripcion;
            $curso->tipo = $request->tipo;
            $curso->creditos = $request->creditos;
            $curso->horas_semestrales = $request->horas_semestrales;
            $curso->orden = $request->orden;
            $curso->semestre_id = $request->semestre_id;
            $curso->id_modulo = $request->id_modulo;
            $curso->save();

            // Registrar la relación con los planes de estudio
            foreach ($request->planes_estudio as $planEstudioId) {
                DB::table('cursos_plan_estudio')->insert([
                    'curso_id' => $curso->id,
                    'plan_estudio_id' => $planEstudioId,
                ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Curso registrado con éxito.',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al registrar el curso: ' . $e->getMessage(),
            ]);
        }
    }

    public function store(Request $request,\App\Services\MoodleService $moodle) 
    {
        $request->validate([
            'nombre' => 'required|string|max:100',
            'descripcion' => 'nullable|string',
            'tipo' => 'required|string',
            'creditos' => 'nullable|string|max:3',
            'horas_semestrales' => 'nullable|integer',
            'orden' => 'nullable|integer',
            'semestre_id' => 'required|exists:semestres,id',
            'id_modulo' => 'required|exists:modulos_formativos,id_modulo',
            'planes_estudio' => 'required|array',
            'planes_estudio.*' => 'exists:planes_estudio,id',
        ]);

        try {

            // 1) CREAR CURSO EN LARAVEL
            $curso = Curso::create([
                'nombre' => $request->nombre,
                'descripcion' => $request->descripcion,
                'tipo' => $request->tipo,
                'creditos' => $request->creditos,
                'horas_semestrales' => $request->horas_semestrales,
                'orden' => $request->orden,
                'semestre_id' => $request->semestre_id,
                'id_modulo' => $request->id_modulo,
            ]);

            // 2) RELACIONAR CON PLANES DE ESTUDIO
            foreach ($request->planes_estudio as $planEstudioId) {
                DB::table('cursos_plan_estudio')->insert([
                    'curso_id' => $curso->id,
                    'plan_estudio_id' => $planEstudioId,
                ]);
            }

            // ==============================
            // 3) CREAR CURSO EN MOODLE
            // ==============================

            // Tomamos SOLO el primer plan (luego refinamos si lo deseas)
            $primerPlan = \App\Models\PlanEstudio::find($request->planes_estudio[0]);

            if ($primerPlan && $primerPlan->moodle_category_id) {

                $shortname = strtoupper(substr($curso->nombre,0,3)) . "_C" . $curso->id;

                $respuestaMoodle = $moodle->crearCurso(
                    $curso->nombre,       // fullname
                    $shortname,           // shortname
                    $primerPlan->moodle_category_id,  // categoría de la carrera
                    $curso->descripcion
                );

                // GUARDAR ID CURSO MOODLE SI SE CREÓ BIEN
                if (isset($respuestaMoodle[0]['id'])) {
                    $curso->moodle_course_id = $respuestaMoodle[0]['id'];
                    $curso->save();
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Curso registrado con éxito.',
                'curso' => $curso,
                'moodle' => $respuestaMoodle ?? null
            ]);

        } catch (\Exception $e) {

            return response()->json([
                'success' => false,
                'message' => 'Error al registrar el curso: ' . $e->getMessage(),
            ]);
        }
    }


    // Mostrar el formulario para editar un curso
    public function edit($id)
    {
        // Obtener el curso a editar junto con sus relaciones
        $curso = Curso::with('modulo', 'semestre', 'planesEstudio')->findOrFail($id);

        // Obtener los planes de estudio y semestres disponibles
        $planesEstudio = PlanEstudio::all();
        $semestres = Semestre::all();

        // Obtener los módulos asociados al plan de estudio del curso
        $modulos = ModuloFormativo::where('id_plan_estudio', $curso->modulo->id_plan_estudio)->get();

        return view('cursos.edit', compact('curso', 'planesEstudio', 'semestres', 'modulos'));
    }


    // Actualizar un curso existente
    public function update(Request $request, $id)
{
    DB::beginTransaction(); // Inicia la transacción

    try {
        $curso = Curso::findOrFail($id);

        // Actualizar el curso
        $curso->nombre = $request->input('nombre');
        $curso->tipo = $request->input('tipo');
        $curso->descripcion = $request->input('descripcion');
        $curso->creditos = $request->input('creditos');
        $curso->horas_semestrales = $request->input('horas_semestrales');
        $curso->orden = $request->input('orden');
        $curso->semestre_id = $request->input('semestre_id');
        $curso->id_modulo = $request->input('id_modulo');
        $curso->save();

        // Eliminar las relaciones antiguas con los planes de estudio
        $curso->planesEstudio()->detach();

        // Insertar la nueva relación con el/los planes de estudio seleccionados
        $curso->planesEstudio()->attach($request->input('planes_estudio'));

        DB::commit(); // Commit de la transacción

        return response()->json([
            'success' => true,
            'message' => 'Curso actualizado con éxito.'
        ]);
    } catch (\Exception $e) {
        DB::rollback(); // Si ocurre un error, revertir cambios

        return response()->json([
            'success' => false,
            'message' => 'Hubo un error al actualizar el curso. Por favor, intente nuevamente.'
        ]);
    }
}


    // Eliminar un curso
    public function destroy($id)
    {
        // Obtener el curso y eliminarlo
        $curso = Curso::findOrFail($id);
        $curso->delete();

        return redirect()->route('cursos.index')->with('success', 'Curso eliminado correctamente.');
    }

    public function modulosPorPlan($planId)
{
    $modulos = ModuloFormativo::where('id_plan_estudio', $planId)->get(['id_modulo', 'nombre']);
    return response()->json($modulos);
}

}
